# SPDX-FileCopyrightText: 2015 Pratik Solanki (Draguu)
#
# SPDX-License-Identifier: GPL-2.0-or-later

bl_info = {
    "name": "Dynamic Sky",
    "author": "Pratik Solanki (Updated for Blender 5.x)",
    "version": (1, 5, 0),
    "blender": (4, 2, 0),
    "location": "View3D > Sidebar > Create Tab",
    "description": "Creates Dynamic Sky for Cycles (Bulletproof Linking)",
    "warning": "",
    "doc_url": "{BLENDER_MANUAL_URL}/addons/lighting/dynamic_sky.html",
    "category": "Lighting",
}

import bpy
from bpy.props import StringProperty
from bpy.types import Operator, Panel


def error_handlers(self, error, reports="ERROR"):
    if self and reports:
        self.report({'WARNING'}, str(reports) + " (See Console)")
    print("\n[Dynamic Sky] Error: {}\n".format(error))


def check_world_name(name_id="Dynamic"):
    name_list = []
    suffix = 1
    try:
        name_list = [world.name for world in bpy.data.worlds if name_id in world.name]
        new_name = "{}_{}".format(name_id, len(name_list) + suffix)
        if new_name in name_list:
            test_num = []
            from re import findall
            for words in name_list:
                test_num.append(findall(r"\d+", words))

            suffix += max([int(l[-1]) for l in test_num])
            new_name = "{}_{}".format(name_id, suffix)
        return new_name
    except Exception as e:
        error_handlers(False, e)
        pass
    return name_id


class dsky(Operator):
    bl_idname = "sky.dyn"
    bl_label = "Make a Procedural sky"
    bl_description = "Make a Procedural Sky (Bulletproof Linking)"

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        try:
            get_name = check_world_name()
            context.scene.dynamic_sky_name = get_name
            if context.scene.render.engine != 'CYCLES':
                bpy.context.scene.render.engine = 'CYCLES'

            world = bpy.data.worlds.new(get_name)
            world.use_nodes = True
            context.scene.world = world

            nt = world.node_tree
            nt.nodes.clear()

            # --- BULLETPROOF LINKER ---
            def ntl(input_socket, output_socket):
                try:
                    nt.links.new(input_socket, output_socket)
                except Exception as e:
                    print(f"Link Error (Ignored): {e}")

            # Helper to get socket by name or index safely
            def get_sock(node, key, is_input=True):
                sockets = node.inputs if is_input else node.outputs
                if isinstance(key, str):
                    try:
                        return sockets[key]
                    except:
                        # Fallback for common name mismatches
                        if key == 'Color' and 'Image' in sockets: return sockets['Image']
                        if key == 'Vector' and 'Normal' in sockets: return sockets['Normal']
                        return sockets[0] # Desperate fallback
                return sockets[key]

            # --- NODE CREATION ---

            # 1. Output System
            world_out = nt.nodes.new(type="ShaderNodeOutputWorld")
            world_out.location = (7200, 360)
            
            bg = nt.nodes.new(type="ShaderNodeBackground")
            bg.name = "Scene_Brightness"
            bg.inputs[0].default_value[:3] = (0.5, .1, 0.6)
            bg.inputs[1].default_value = 1
            bg.location = (6800, 360)

            # 2. Coordinates & Mapping
            tcor = nt.nodes.new(type="ShaderNodeTexCoord")
            tcor.location = (250, 1000)

            map1 = nt.nodes.new(type="ShaderNodeMapping")
            map1.vector_type = 'NORMAL'
            map1.location = (800, 730)

            nor = nt.nodes.new(type="ShaderNodeNormal")
            nor.name = "Sky_normal"
            nor.location = (1250, 685)

            # 3. Color Ramps
            cr1 = nt.nodes.new(type="ShaderNodeValToRGB")
            cr1.color_ramp.elements[0].position = 0.969
            cr1.color_ramp.interpolation = 'EASE'
            cr1.location = (1700, 415)
            
            cr2 = nt.nodes.new(type="ShaderNodeValToRGB")
            cr2.color_ramp.elements[0].position = 0.991
            cr2.color_ramp.elements[1].position = 1
            cr2.color_ramp.interpolation = 'EASE'
            cr2.location = (2200, 415)
            
            cr3 = nt.nodes.new(type="ShaderNodeValToRGB")
            cr3.color_ramp.elements[0].position = 0.779
            cr3.color_ramp.elements[1].position = 1
            cr3.color_ramp.interpolation = 'EASE'
            cr3.location = (2200, 415)

            # 4. Math Nodes
            mat1 = nt.nodes.new(type="ShaderNodeMath")
            mat1.operation = 'MULTIPLY'
            mat1.inputs[1].default_value = 0.2
            mat1.location = (2200, 685)
            
            mat2 = nt.nodes.new(type="ShaderNodeMath")
            mat2.operation = 'MULTIPLY'
            mat2.inputs[1].default_value = 2
            mat2.location = (3300, 685)
            
            mat3 = nt.nodes.new(type="ShaderNodeMath")
            mat3.operation = 'MULTIPLY'
            mat3.inputs[1].default_value = 40.9
            mat3.location = (2750, 415)
            
            mat4 = nt.nodes.new(type="ShaderNodeMath")
            mat4.operation = 'SUBTRACT'
            mat4.inputs[1].default_value = 1
            mat4.location = (3300, 415)

            # 5. Mix Nodes (Typ RGBA - Index 6=A, 7=B, 2=Result)
            soft = nt.nodes.new(type="ShaderNodeMix")
            soft.data_type = 'RGBA'
            soft.name = "Soft_hard"
            soft.location = (3850, 550)
            soft.inputs[0].default_value = 1
            
            soft_1 = nt.nodes.new(type="ShaderNodeMix")
            soft_1.data_type = 'RGBA'
            soft_1.location = (3850, 185)
            soft_1.inputs[0].default_value = 0.466

            mix1 = nt.nodes.new(type="ShaderNodeMix")
            mix1.data_type = 'RGBA'
            mix1.blend_type = 'MULTIPLY'
            mix1.inputs[0].default_value = 1
            mix1.location = (4350, 630)
            
            mix1_1 = nt.nodes.new(type="ShaderNodeMix")
            mix1_1.data_type = 'RGBA'
            mix1_1.blend_type = 'MULTIPLY'
            mix1_1.inputs[0].default_value = 1
            mix1_1.location = (4350, 90)

            mix2 = nt.nodes.new(type="ShaderNodeMix")
            mix2.data_type = 'RGBA'
            mix2.location = (4800, 610)
            mix2.inputs[6].default_value = (0, 0, 0, 1)
            mix2.inputs[7].default_value = (32, 22, 14, 200)
            
            mix2_1 = nt.nodes.new(type="ShaderNodeMix")
            mix2_1.data_type = 'RGBA'
            mix2_1.location = (5150, 270)
            mix2_1.inputs[6].default_value = (0, 0, 0, 1)
            mix2_1.inputs[7].default_value = (1, 0.820, 0.650, 1)

            gam = nt.nodes.new(type="ShaderNodeGamma")
            gam.inputs[1].default_value = 2.3
            gam.location = (5150, 610)

            gam2 = nt.nodes.new(type="ShaderNodeGamma")
            gam2.name = "Sun_value"
            gam2.inputs[1].default_value = 1
            gam2.location = (5550, 610)

            gam3 = nt.nodes.new(type="ShaderNodeGamma")
            gam3.name = "Shadow_color_saturation"
            gam3.inputs[1].default_value = 1
            gam3.location = (5550, 880)

            sunopa = nt.nodes.new(type="ShaderNodeMix")
            sunopa.data_type = 'RGBA'
            sunopa.blend_type = 'ADD'
            sunopa.inputs[0].default_value = 1
            sunopa.location = (5950, 610)
            
            sunopa_1 = nt.nodes.new(type="ShaderNodeMix")
            sunopa_1.data_type = 'RGBA'
            sunopa_1.blend_type = 'ADD'
            sunopa_1.inputs[0].default_value = 1
            sunopa_1.location = (5550, 340)

            combine = nt.nodes.new(type="ShaderNodeMix")
            combine.data_type = 'RGBA'
            combine.location = (6350, 360)
            
            lp = nt.nodes.new(type="ShaderNodeLightPath")
            lp.location = (5950, 130)

            # 6. Noise & Clouds
            map2 = nt.nodes.new(type="ShaderNodeMapping")
            map2.inputs['Scale'].default_value[:] = (1.5, 1.5, 6.0)
            map2.location = (2200, 1510)

            n1 = nt.nodes.new(type="ShaderNodeTexNoise")
            n1.inputs['Scale'].default_value = 3.8
            n1.inputs['Detail'].default_value = 2.4
            n1.inputs['Distortion'].default_value = 0.5
            n1.location = (2750, 1780)

            n2 = nt.nodes.new(type="ShaderNodeTexNoise")
            n2.inputs['Scale'].default_value = 2.0
            n2.inputs['Detail'].default_value = 10
            n2.inputs['Distortion'].default_value = 0.2
            n2.location = (2750, 1510)

            sc1 = nt.nodes.new(type="ShaderNodeValToRGB")
            sc1.location = (3300, 1780)
            sc1.color_ramp.elements[0].position = 0.408
            sc1.color_ramp.elements[1].position = 0.649
            
            sc2 = nt.nodes.new(type="ShaderNodeValToRGB")
            sc2.location = (3300, 1510)
            sc2.color_ramp.elements[0].position = 0.408
            sc2.color_ramp.elements[1].position = 0.576

            sc3 = nt.nodes.new(type="ShaderNodeValToRGB")
            sc3.location = (3850, 820)
            sc3.color_ramp.elements[0].position = 0.027
            sc3.color_ramp.elements[0].color = (0.419, 0.419, 0.419, 0.419)
            sc3.color_ramp.elements[1].position = 0.160
            sc3.color_ramp.elements[1].color = (1, 1, 1, 1)
            sc3.color_ramp.elements.new(0.435)

            sc3_1 = nt.nodes.new(type="ShaderNodeValToRGB")
            sc3_1.location = (4350, 1360)
            sc3_1.color_ramp.elements[0].position = 0.0
            sc3_1.color_ramp.elements[0].color = (0, 0, 0, 0)
            sc3_1.color_ramp.elements[1].position = 0.187
            sc3_1.color_ramp.elements[1].color = (1, 1, 1, 1)
            sc3_1.color_ramp.elements.new(0.435)

            sc4 = nt.nodes.new(type="ShaderNodeValToRGB")
            sc4.location = (3850, 1090)
            sc4.color_ramp.elements[0].position = 0.0
            sc4.color_ramp.elements[0].color = (1, 1, 0.917412, 1)
            sc4.color_ramp.elements[1].position = 0.469
            sc4.color_ramp.elements[1].color = (0, 0, 0, 1)

            smix1 = nt.nodes.new(type="ShaderNodeMix")
            smix1.data_type = 'RGBA'
            smix1.location = (3850, 1550)
            smix1.name = "Cloud_color"
            smix1.inputs[6].default_value = (1, 1, 1, 1)
            smix1.inputs[7].default_value = (0, 0, 0, 1)
            
            smix2 = nt.nodes.new(type="ShaderNodeMix")
            smix2.data_type = 'RGBA'
            smix2.location = (4350, 1630)
            smix2.name = "Cloud_density"
            smix2.blend_type = 'MULTIPLY'
            smix2.inputs[0].default_value = 0.267
            
            smix2_1 = nt.nodes.new(type="ShaderNodeMix")
            smix2_1.data_type = 'RGBA'
            smix2_1.location = (4800, 1360)
            smix2_1.blend_type = 'MULTIPLY'
            smix2_1.inputs[0].default_value = 1

            smix3 = nt.nodes.new(type="ShaderNodeMix")
            smix3.data_type = 'RGBA'
            smix3.location = (4350, 1090)
            smix3.name = "Sky_and_Horizon_colors"
            smix3.inputs[6].default_value = (0.434, 0.838, 1, 1)
            smix3.inputs[7].default_value = (0.962, 0.822, 0.822, 1)

            smix4 = nt.nodes.new(type="ShaderNodeMix")
            smix4.data_type = 'RGBA'
            smix4.location = (4800, 880)
            smix4.blend_type = 'MULTIPLY'
            smix4.inputs[0].default_value = 1

            smix5 = nt.nodes.new(type="ShaderNodeMix")
            smix5.data_type = 'RGBA'
            smix5.name = "Cloud_opacity"
            smix5.location = (5150, 880)
            smix5.blend_type = 'SCREEN'
            smix5.inputs[0].default_value = 1

            # 7. Color Split/Combine
            srgb = nt.nodes.new(type="ShaderNodeSeparateColor")
            srgb.location = (800, 1370)
            
            aniadd = nt.nodes.new(type="ShaderNodeMath")
            aniadd.location = (1250, 1235)
            
            crgb = nt.nodes.new(type="ShaderNodeCombineColor")
            crgb.location = (1700, 1510)
            
            sunrgb = nt.nodes.new(type="ShaderNodeMix")
            sunrgb.data_type = 'RGBA'
            sunrgb.name = "Sun_color"
            sunrgb.blend_type = 'MULTIPLY'
            sunrgb.inputs[6].default_value = (32, 30, 30, 200)
            sunrgb.inputs[0].default_value = 0 
            sunrgb.location = (4350, 360)

            skynor = nt.nodes.new(type="ShaderNodeNormal")
            skynor.location = (3300, 1070)

            # --- LINKING (With Safety Nets) ---

            # World Output
            ntl(bg.outputs[0], world_out.inputs[0])
            ntl(combine.outputs[2], bg.inputs[0])

            # Ramps & Math
            ntl(mat2.inputs[0], mat1.outputs[0])
            ntl(mat4.inputs[0], mat3.outputs[0])
            ntl(mat1.inputs[0], cr3.outputs[0])
            ntl(mat3.inputs[0], cr2.outputs[0])

            # Mixes (Explicit Index 6/7 for Color A/B)
            ntl(soft.inputs[6], mat2.outputs[0])
            ntl(soft.inputs[7], mat4.outputs[0])
            ntl(soft_1.inputs[6], mat2.outputs[0])
            ntl(soft_1.inputs[7], cr2.outputs[0])

            ntl(mix1.inputs[6], soft.outputs[2])
            ntl(mix1_1.inputs[6], soft_1.outputs[2])
            ntl(mix2.inputs[0], mix1.outputs[2])
            ntl(mix2_1.inputs[0], mix1_1.outputs[2])
            
            ntl(mix2.inputs[7], sunrgb.outputs[2])
            
            ntl(combine.inputs[6], sunopa.outputs[2])
            ntl(combine.inputs[7], sunopa_1.outputs[2])
            
            # Light Path (Output 0 -> Mix Fac 0)
            ntl(combine.inputs[0], lp.outputs[0])

            ntl(gam2.inputs[0], gam.outputs[0])
            ntl(gam.inputs[0], mix2.outputs[2])

            # Noise Logic (Safe Names)
            ntl(n2.inputs['Vector'], map2.outputs['Vector'])
            ntl(n1.inputs['Vector'], map2.outputs['Vector'])
            ntl(sc1.inputs[0], n1.outputs['Fac'])
            ntl(sc2.inputs[0], n2.outputs['Fac'])

            ntl(smix1.inputs[0], sc2.outputs[0])
            ntl(smix2.inputs[6], smix1.outputs[2])
            ntl(smix2.inputs[7], sc1.outputs[0])
            ntl(smix2_1.inputs[7], sc3_1.outputs[0])
            
            ntl(smix3.inputs[0], sc4.outputs[0])
            ntl(smix4.inputs[7], smix3.outputs[2])
            ntl(smix4.inputs[6], sc3.outputs[0])
            ntl(smix5.inputs[6], smix4.outputs[2])
            ntl(smix2_1.inputs[6], smix2.outputs[2])
            ntl(smix5.inputs[7], smix2_1.outputs[2])
            
            ntl(sunopa.inputs[6], gam3.outputs[0])
            ntl(gam3.inputs[0], smix5.outputs[2])
            ntl(mix1.inputs[7], sc3.outputs[0])
            ntl(sunopa.inputs[7], gam2.outputs[0])
            ntl(sunopa_1.inputs[6], smix5.outputs[2])
            ntl(sunopa_1.inputs[7], mix2_1.outputs[2])

            # Sky Normal & Coordinates
            ntl(sc3.inputs[0], skynor.outputs['Dot'])
            ntl(sc4.inputs[0], skynor.outputs['Dot'])
            ntl(sc3_1.inputs[0], skynor.outputs['Dot'])
            ntl(map2.inputs['Vector'], crgb.outputs[0])
            ntl(skynor.inputs['Normal'], tcor.outputs['Generated'])
            ntl(mix1_1.inputs[7], sc3.outputs[0])

            # Color Logic (Using safe name checks for Separate/Combine)
            # SeparateColor: Out 1=Red, 2=Green, 3=Blue (Usually)
            # We use fallback logic via get_sock just in case names changed.
            
            # TexCoord -> SeparateColor
            ntl(srgb.inputs[0], tcor.outputs['Generated'])
            
            # Red Channel Logic
            # srgb Red -> Math
            ntl(aniadd.inputs[1], get_sock(srgb, 'Red', False)) 
            # Math -> Combine Red
            ntl(get_sock(crgb, 'Red'), aniadd.outputs[0]) 
            
            # Green/Blue Pass-through
            ntl(get_sock(crgb, 'Green'), get_sock(srgb, 'Green', False))
            ntl(get_sock(crgb, 'Blue'), get_sock(srgb, 'Blue', False))

            # Sun Direction
            ntl(cr1.inputs[0], nor.outputs['Dot'])
            ntl(cr2.inputs[0], cr1.outputs[0])
            ntl(cr3.inputs[0], nor.outputs['Dot'])
            
            # Ensure Map -> Normal connection
            ntl(nor.inputs['Normal'], map1.outputs['Vector'])
            ntl(map1.inputs['Vector'], tcor.outputs['Generated'])

        except Exception as e:
            error_handlers(self, e, "Failed to create Sky")
            import traceback
            traceback.print_exc()
            return {"CANCELLED"}

        return {'FINISHED'}


def draw_world_settings(col, context):
    get_world = context.scene.world
    stored_name = context.scene.dynamic_sky_name
    get_world_keys = bpy.data.worlds.keys()

    if stored_name not in get_world_keys or len(get_world_keys) < 1:
        col.label(text="World not found", icon="INFO")
        return

    elif not (get_world and get_world.name == stored_name):
        col.label(text="Please select the World", icon="INFO")
        col.label(text=stored_name, icon="BLANK1")
        return

    pick_world = bpy.data.worlds[stored_name]
    nodes = pick_world.node_tree.nodes
    
    try:
        col.label(text="World: %s" % stored_name)
        col.separator()

        col.label(text="Scene Control")
        if 'Scene_Brightness' in nodes:
            col.prop(nodes['Scene_Brightness'].inputs[1], "default_value", text="Brightness")
        if 'Shadow_color_saturation' in nodes:
            col.prop(nodes['Shadow_color_saturation'].inputs[1], "default_value", text="Shadow Saturation")

        col.label(text="Sky Control")
        if 'Sky_and_Horizon_colors' in nodes:
            col.prop(nodes['Sky_and_Horizon_colors'].inputs[6], "default_value", text="Sky Color")
            col.prop(nodes['Sky_and_Horizon_colors'].inputs[7], "default_value", text="Horizon Color")
        
        if 'Cloud_color' in nodes:
            col.prop(nodes['Cloud_color'].inputs[6], "default_value", text="Cloud Color")
        if 'Cloud_opacity' in nodes:
            col.prop(nodes['Cloud_opacity'].inputs[0], "default_value", text="Cloud Opacity")
        if 'Cloud_density' in nodes:
            col.prop(nodes['Cloud_density'].inputs[0], "default_value", text="Cloud Density")

        col.label(text="Sun Control")
        if 'Sun_color' in nodes:
            col.prop(nodes['Sun_color'].inputs[6], "default_value", text="Sun Color")
        if 'Sun_value' in nodes:
            col.prop(nodes['Sun_value'].inputs[1], "default_value", text="Sun Value")
        if 'Soft_hard' in nodes:
            col.prop(nodes['Soft_hard'].inputs[0], "default_value", text="Soft/Hard")
        if 'Sky_normal' in nodes:
            # Controls the Output socket default value (Internal sphere widget)
            col.prop(nodes['Sky_normal'].outputs[0], "default_value", text="Sun Direction")

    except Exception as e:
        col.label(text="UI Error: " + str(e), icon="ERROR")

class Dynapanel(Panel):
    bl_label = "Dynamic sky"
    bl_idname = "DYNSKY_PT_tools"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_context = "objectmode"
    bl_category = "Create"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        layout.operator("sky.dyn", text="Create", icon='MAT_SPHERE_SKY')
        col = layout.column()
        draw_world_settings(col, context)

def register():
    bpy.utils.register_class(Dynapanel)
    bpy.utils.register_class(dsky)
    bpy.types.Scene.dynamic_sky_name = StringProperty(name="", default="Dynamic")

def unregister():
    bpy.utils.unregister_class(Dynapanel)
    bpy.utils.unregister_class(dsky)
    del bpy.types.Scene.dynamic_sky_name

if __name__ == "__main__":
    register()