/*
	Loads, keeps and saves all of my settings.
	Mostly static methods.
	
	Some code from artofillusion.keystroke.KeystrokeManager
	used as a base.
	
	.dm'TroY., October 2007
 */

package troy.generic;

import artofillusion.Plugin;
import artofillusion.ModellingApp;
import artofillusion.SafeFileOutputStream;

import java.io.*;
import java.util.Hashtable;
import java.util.Enumeration;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;

import org.w3c.dom.*;

/**
 * Class to load, keep and save all of my settings. Mostly static except for auto-load/-save.
 */
public class TroySettings implements Plugin
{
	private static final File SETTINGSFILE = new File(ModellingApp.getPreferences().getPreferencesDirectory(), "TroyTools.xml");
	
	private static Hashtable<String,Hashtable<String,String>> set = new Hashtable<String,Hashtable<String,String>>();

	/**
	 * for debug purposes: dump the hashtable to console/log
	 */
	private static void dumpSet()
	{
		// ########################################################################
		
		String out = "START\n";
		
		Enumeration x = set.keys();
		while (x.hasMoreElements())
		{
			String section = (String)x.nextElement();
			Hashtable<String,String> inner = set.get(section);
			
			out += "Section: |" + section + "|\n";
			
			Enumeration y = inner.keys();
			while (y.hasMoreElements())
			{
				String key   = (String)y.nextElement();
				String value = inner.get(key);
				
				out += "\t|" + key + "| = |" + value + "|\n";
			}
		}
		
		out += "EOF";
		
		System.out.println(out);
		
		// ########################################################################
	}
	
	/**
	 * set a specific item in the specified section. if the section does not
	 * yet exist, it will be automatically created.
	 */
	public static void set(String section, String key, String value)
	{
		if (set.get(section) == null)
		{
			set.put(section, new Hashtable<String,String>());
		}
		
		set.get(section).put(key, value);
		
		//dumpSet();
	}
	
	/**
	 * get an item out of the specified section - returns AN EMPTY STRING if not found!
	 */
	public static String get(String section, String key)
	{
		Hashtable<String,String> inner = set.get(section);
		
		if (inner == null)
			return "";
		
		String out = inner.get(key);
		if (out == null)
			return "";
		else
			return out;
	}
	
	/**
	 * load settings
	 */
	public static void load()
	{
		try
		{
			File inputFile = SETTINGSFILE;
			InputStream in;
			if (inputFile.exists())
			{
				in = new BufferedInputStream(new FileInputStream(inputFile));
				readFromXML(in);
				in.close();
			}
			else
			{
				System.out.println("TroySettings: " + SETTINGSFILE + " does not exist, loading default values.");
				
				TroySettings.set("startup", "boxMaximize", "1");
				TroySettings.set("startup", "boxShow", "1");
				TroySettings.set("startup", "boxSnap", "1");
				TroySettings.set("startup", "boxAmbient", "1");
				TroySettings.set("startup", "boxLight", "1");
				TroySettings.set("startup", "boxCam", "1");
				
				TroySettings.set("autobackup", "boxActive", "0");
				TroySettings.set("autobackup", "textLevels", "10");
			}
		}
		catch (Exception ex)
		{
			System.out.print("TroySettings: " + SETTINGSFILE + " could not be read, StackTrace:\n\n");
			ex.printStackTrace();
		}
	}
	
	/**
	 * if the inputstream for loading has been verified, read from it and
	 * parse xml
	 */
	private static void readFromXML(InputStream in) throws Exception
	{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document doc = builder.parse(in);

		NodeList sections = doc.getFirstChild().getChildNodes();
		
		for (int i = 0; i < sections.getLength(); i++)
		{
			Node sectionNode = sections.item(i);
			
			if (sectionNode.hasChildNodes())
			{
				String strSECTION = sectionNode.getNodeName();

				set.put(strSECTION, new Hashtable<String,String>());
				Hashtable<String,String> thisSection = set.get(strSECTION);
				//System.out.println("Section: " + strSECTION);
				
				NodeList inner = sectionNode.getChildNodes();
				
				for (int j = 0; j < inner.getLength(); j++)
				{
					Node keyvalNode = inner.item(j);
				
					if (keyvalNode.hasAttributes())
					{
						String strKEY = keyvalNode.getNodeName();
						String strVAL = keyvalNode.getAttributes().getNamedItem("value").getNodeValue();
						//System.out.println("\t" + strKEY + " = " + strVAL);
						
						thisSection.put(strKEY, strVAL);
					}
				}
			}
		}
		
		//dumpSet();
	}
	
	/**
	 * save settings
	 */
	public static void save()
	{
		try
		{
			// save settings
			File outFile = SETTINGSFILE;
					
			System.out.println("TroySettings: Saving settings to " + outFile.getAbsolutePath());

			// build xml document
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document doc = builder.newDocument();
			
			Element docroot = doc.createElement("settings");
			doc.appendChild(docroot);
			
			Enumeration e = set.keys();
			while (e.hasMoreElements())
			{
				String section = (String)e.nextElement();
				Hashtable<String,String> inner = set.get(section);
				
				// add a root element
				Element root = doc.createElement(section);
				
				// append this to our document
				docroot.appendChild(root);
				
				Enumeration f = inner.keys();
				while (f.hasMoreElements())
				{
					String key   = (String)f.nextElement();
					String value = inner.get(key);
					
					Element recordElement = doc.createElement(key);
					recordElement.setAttribute("value", value);
					root.appendChild(recordElement);
				}
			}
			
			// write to disk
			OutputStream out = new BufferedOutputStream(new SafeFileOutputStream(outFile, SafeFileOutputStream.OVERWRITE));
			DOMSource source = new DOMSource(doc);
			StreamResult result = new StreamResult(out);
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.transform(source, result);
			out.close();
			
			System.out.println("TroySettings: Saving ok.");
		}
		catch (Exception e)
		{
			System.out.println("TroySettings: Exception while saving, StackTrace:\n\n");
			e.printStackTrace();
		}
	}
	
	
	/**
	 * process AoI messages: auto-load and auto-save.
	 */
	public void processMessage(int message, java.lang.Object[] args)
	{
		if (message == APPLICATION_STARTING)
		{
			TroySettings.load();
		}
		else if (message == APPLICATION_STOPPING)
		{
			TroySettings.save();
		}
	}
}
