package troy.advcurves;

import artofillusion.*;
import artofillusion.math.*;
import artofillusion.object.*;
import artofillusion.ui.*;
import buoy.event.*;
import java.awt.*;
import java.util.*;

/**
 * Add new points at the end or beginning of a curve
 * @author TroY
 */
public class ExtendCurveTool extends MeshEditingTool
{
	/** Common constructor, init tool button */
	public ExtendCurveTool(EditingWindow fr, MeshEditController controller)
	{
		super(fr, controller);
		initButton("advcurves:extendCurve");
	}
	
	/** Tell AoI which clicks we want to catch */
	@Override
	public int whichClicks()
	{
		return ALL_CLICKS;
	}
	
	/** This tool is able to change the current selection */
	@Override
	public boolean allowSelectionChanges()
	{
		return true;
	}
	
	/** What to do when the user activates the tool: Set the window's help text */
	@Override
	public void activate()
	{
		super.activate();
		theWindow.setHelpText(Translate.text("advcurves:extendCurveTool.helpText"));
	}
	
	/** Return tool tip */
	@Override
	public String getToolTipText()
	{
		return Translate.text("advcurves:extendCurveTool.tipText");
	}
	
	/** User has clicked somewhere in the scene - we are to extend the curve now */
	@Override
	public void mousePressed(WidgetMouseEvent e, ViewerCanvas view)
	{
		boolean[] selected = controller.getSelection();
		Curve     obj      = (controller.getObject().getObject() instanceof Curve ? (Curve)(controller.getObject().getObject()) : null);
		
		if (obj != null)
		{
			int selCount = 0;
			Vector<Integer> which = new Vector<Integer>();
			for (int i = 0; i < selected.length; i++)
			{
				if (selected[i])
				{
					selCount++;
					which.add(i);
				}
			}
			
			// beginning or end
			if (selCount == 1 && (selected[0] || selected[selected.length - 1]))
			{
				// undo record!
				theWindow.setUndoRecord(new UndoRecord(theWindow, false, UndoRecord.COPY_OBJECT, new Object [] {obj, obj.duplicate()}));
				
				// use the camera to transform screen coordinates to world coordinates
				Camera cam = view.getCamera();
				
				// calc distance from cam to the new point - we're doing it the
				// same way as it's done with bones
				// first, geht the depth of the currently selected point ...
				double distance = cam.getWorldToView().timesZ(obj.getVertices()[which.get(0)].r);
				// ... then look for the clicked point in this depth
				Vec3 newPoint = Snapper.snap(cam.convertScreenToWorld(e.getPoint(), distance), view);
				
				MeshVertex[] origV = obj.getVertices();
				float[] origF = obj.getSmoothness();
				
				Vec3[]  v = new  Vec3[origV.length + 1];
				float[] f = new float[origF.length + 1];
				
				boolean[] s = new boolean[v.length];
				
				// extended at the end:
				if (selected[selected.length - 1])
				{
				
					for (int i = 0; i < origV.length; i++)
					{
						v[i] = origV[i].r;
						f[i] = origF[i];
						s[i] = false;
					}
					
					v[v.length - 1] = newPoint;
					f[f.length - 1] = 1.0f;
					s[s.length - 1] = true;
				
				}
				// extended at the beginning:
				else
				{
					for (int i = 0; i < origV.length; i++)
					{
						v[i + 1] = origV[i].r;
						f[i + 1] = origF[i];
						s[i + 1] = false;
					}
					
					v[0] = newPoint;
					f[0] = 1.0f;
					s[0] = true;
				}
				
				// update object and editor window:
				obj.setShape(v, f);
				((CurveEditorWindow)controller).setSelection(s);
				
				//System.out.println("Extension done (" + obj + "). " + v.length + "," + f.length + "," + s.length);
			}
			// in the middle of the curve - see above comments for most commands here
			// only allow insertion between first/last if there are only 2 vertices on the curve
			else if (selCount == 2 && (!(selected[0] && selected[selected.length - 1]) || selected.length == 2))
			{
				// we have to make sure that the selected vertices are neighbours.
				if (Math.abs(which.get(0) - which.get(1)) == 1)
				{
					theWindow.setUndoRecord(new UndoRecord(theWindow, false, UndoRecord.COPY_OBJECT, new Object [] {obj, obj.duplicate()}));

					Camera cam = view.getCamera();
					
					double distance0 = cam.getWorldToView().timesZ(obj.getVertices()[which.get(0)].r);
					double distance1 = cam.getWorldToView().timesZ(obj.getVertices()[which.get(1)].r);
					double distance = (distance0 + distance1) / 2.0;
					
					Vec3 newPoint = Snapper.snap(cam.convertScreenToWorld(e.getPoint(), distance), view);
					
					// new point found, now insert it
					MeshVertex[] origV = obj.getVertices();
					float[] origF = obj.getSmoothness();
					
					Vec3[]  v = new  Vec3[origV.length + 1];
					float[] f = new float[origF.length + 1];
					
					boolean[] s = new boolean[v.length];
					
					int npos = 0;
					int i    = 0;
					while (i < origV.length)
					{
						v[npos] = origV[i].r;
						f[npos] = origF[i];
						s[npos] = false;
						
						if (i == which.get(0))
						{
							npos++;
							v[npos] = newPoint;
							f[npos] = 1.0f;
							s[npos] = true;
						}
						
						i++;
						npos++;
					}
					
					// update object and editor window:
					obj.setShape(v, f);
					((CurveEditorWindow)controller).setSelection(s);
				}
			}
		}
	}
}
